<?php
require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/auth.php';
require_login();
$u = current_user();
require __DIR__ . '/_header.php';
?>
<div class="row g-3">
  <div class="col-12 col-lg-7">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="d-flex align-items-center justify-content-between">
          <h1 class="h5 mb-0">Scanner QR</h1>
          <div class="small text-secondary">
            Login: <b><?= h($u['username']) ?></b> (<?= h($u['role']) ?>)
            • <a href="/public/logout.php">Logout</a>
          </div>
        </div>

        <div class="mt-3 d-flex gap-2 flex-wrap">
          <button class="btn btn-primary" id="modeCheckin">Mode CHECK-IN</button>
          <button class="btn btn-outline-primary" id="modeSouvenir">Mode SOUVENIR</button>
          <button class="btn btn-outline-secondary" id="btnSwitchCam">Switch Camera</button>
        </div>

        <div class="mt-3 scanner-box p-2">
          <div id="reader" style="width:100%"></div>
        </div>

        <div class="mt-3" id="resultArea"></div>

        <div class="small text-secondary mt-3">
          Tips: gunakan koneksi stabil. iPhone wajib akses lewat <b>HTTPS</b>.
        </div>
      </div>
    </div>
  </div>

  <div class="col-12 col-lg-5">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="fw-semibold mb-2">Status</div>
        <div class="alert alert-light border mb-2">
          Mode aktif: <span class="fw-semibold" id="modeLabel">CHECKIN</span>
        </div>
        <div class="alert alert-light border mb-0">
          Terakhir scan: <span class="fw-semibold" id="lastCode">-</span>
        </div>
        <hr>
        <div class="small text-secondary">
          Jika QR berisi link, sistem akan mengambil <code>code</code> dari parameter <code>?code=</code> atau path terakhir.
          Jika QR berisi kode saja, sistem langsung pakai kode tersebut.
        </div>
      </div>
    </div>
  </div>
</div>

<script src="https://unpkg.com/html5-qrcode@2.3.10/html5-qrcode.min.js"></script>
<script>
let mode = "CHECKIN";
let currentCameraId = null;
let cameras = [];
const modeLabel = document.getElementById("modeLabel");
const lastCode = document.getElementById("lastCode");
const resultArea = document.getElementById("resultArea");

function setMode(m) {
  mode = m;
  modeLabel.textContent = m;
  document.getElementById("modeCheckin").className = (m==="CHECKIN") ? "btn btn-primary" : "btn btn-outline-primary";
  document.getElementById("modeSouvenir").className = (m==="SOUVENIR") ? "btn btn-primary" : "btn btn-outline-primary";
}

function extractCode(decodedText) {
  // If it's a URL, try to extract ?code= or last path segment
  try {
    if (decodedText.startsWith("http://") || decodedText.startsWith("https://")) {
      const u = new URL(decodedText);
      const c = u.searchParams.get("code");
      if (c) return c.trim();
      const parts = u.pathname.split("/").filter(Boolean);
      if (parts.length) return parts[parts.length-1].trim();
    }
  } catch (e) {}
  return decodedText.trim();
}

function showResult(ok, title, msg, extra={}) {
  const cls = ok ? "alert alert-success" : "alert alert-danger";
  const badge = ok ? "text-bg-success" : "text-bg-danger";
  const guest = extra.guest ? `<div class="mt-1"><span class="badge ${badge}">Tamu: ${escapeHtml(extra.guest)}</span></div>` : "";
  const meta = extra.meta ? `<div class="small text-secondary mt-2">${escapeHtml(extra.meta)}</div>` : "";
  resultArea.innerHTML = `
    <div class="${cls}">
      <div class="fw-semibold">${escapeHtml(title)}</div>
      <div>${escapeHtml(msg)}</div>
      ${guest}
      ${meta}
    </div>
  `;
  // beep + vibrate (best-effort)
  try { if (ok) navigator.vibrate?.(120); } catch(e) {}
  try {
    if (ok) {
      const ctx = new (window.AudioContext || window.webkitAudioContext)();
      const o = ctx.createOscillator();
      const g = ctx.createGain();
      o.connect(g); g.connect(ctx.destination);
      o.type = "sine"; o.frequency.value = 880;
      g.gain.setValueAtTime(0.001, ctx.currentTime);
      g.gain.exponentialRampToValueAtTime(0.2, ctx.currentTime + 0.01);
      o.start();
      setTimeout(()=>{ o.stop(); ctx.close(); }, 120);
    }
  } catch(e) {}
}

function escapeHtml(s) {
  return String(s).replace(/[&<>"']/g, (m)=>({ "&":"&amp;","<":"&lt;",">":"&gt;",'"':"&quot;","'":"&#39;" }[m]));
}

async function hitApi(code) {
  const url = (mode === "CHECKIN") ? "/public/api/scan_checkin.php" : "/public/api/scan_souvenir.php";
  const resp = await fetch(url, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ code })
  });
  return await resp.json();
}

// Prevent duplicate rapid fires
let lastScanAt = 0;
let lastScannedCode = "";

function onScanSuccess(decodedText) {
  const code = extractCode(decodedText);
  const now = Date.now();
  if (code === "" ) return;
  if (code === lastScannedCode && (now - lastScanAt) < 2000) return;
  lastScannedCode = code;
  lastScanAt = now;

  lastCode.textContent = code;
  showResult(true, "Terbaca", "Memproses...", { meta: "CODE: " + code });

  hitApi(code).then((data) => {
    if (data.ok) showResult(true, data.status || "OK", data.msg || "Berhasil", { guest: data.guest || "" });
    else showResult(false, data.status || "GAGAL", data.msg || "Gagal", { guest: data.guest || "" });
  }).catch((err) => {
    showResult(false, "ERROR", err?.message || "Terjadi kesalahan");
  });
}

const html5QrCode = new Html5Qrcode("reader");

async function startScanner(cameraId=null) {
  const config = { fps: 10, qrbox: { width: 250, height: 250 } };
  await html5QrCode.stop().catch(()=>{});
  await html5QrCode.clear().catch(()=>{});
  await html5QrCode.start(
    { deviceId: cameraId ? { exact: cameraId } : undefined, facingMode: cameraId ? undefined : "environment" },
    config,
    onScanSuccess
  );
}

Html5Qrcode.getCameras().then((devices) => {
  cameras = devices || [];
  if (cameras.length > 0) {
    currentCameraId = cameras[0].id;
  }
  startScanner(currentCameraId);
}).catch((err) => {
  resultArea.innerHTML = `<div class="alert alert-danger">Tidak bisa akses kamera: ${escapeHtml(err)}</div>`;
});

document.getElementById("modeCheckin").addEventListener("click", ()=>setMode("CHECKIN"));
document.getElementById("modeSouvenir").addEventListener("click", ()=>setMode("SOUVENIR"));

document.getElementById("btnSwitchCam").addEventListener("click", async () => {
  if (!cameras || cameras.length < 2) {
    showResult(false, "INFO", "Kamera hanya 1 terdeteksi");
    return;
  }
  const idx = cameras.findIndex(c => c.id === currentCameraId);
  const next = cameras[(idx + 1) % cameras.length];
  currentCameraId = next.id;
  await startScanner(currentCameraId);
  showResult(true, "OK", "Switch camera berhasil");
});
</script>

<?php require __DIR__ . '/_footer.php'; ?>
